﻿using Hims.Api.Models;
using Hims.Api.Utilities;
using Hims.Domain.Services;
using Hims.Infrastructure.Services;
using Hims.Shared.DataFilters;
using Hims.Shared.EntityModels;
using Hims.Shared.Library.Enums;
using Hims.Shared.UserModels;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

namespace Hims.Api.Controllers
{
    /// <inheritdoc />
    [Authorize]
    [Route("api/case-type")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class CaseTypeController : BaseController
    {
        /// <summary>
        /// The idproof services.
        /// </summary>
        private readonly ICaseTypeService caseTypeService;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public CaseTypeController(ICaseTypeService caseTypeService, IAuditLogService auditLogService)
        {
            this.caseTypeService = caseTypeService;
            this.auditLogServices = auditLogService;
        }

        /// <summary>
        /// The Insert Api
        /// </summary>
        /// <param name="model"></param>
        /// <param name="header"></param>
        /// <returns></returns>
        [HttpPost]
        [Route("insert")]
        public async Task<ActionResult> InsertAsync([FromBody] CaseTypeModel model, [FromHeader] LocationHeader header)
        {
            model = (CaseTypeModel)EmptyFilter.Handler(model);
            var response = await this.caseTypeService.InsertAsync(model);
            if (response > 0)
            {
                var auditLog = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.CaseType,
                    LogDate = DateTime.Now,
                    LogFrom = (short)model.LoginRoleId,
                    LogDescription = $"{model.CreatedByName} has added <b>Case Type Name</b> of <strong>{model.CaseTypeName}</strong> successfully.",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLog);
            }

            return this.Success(response);
        }

        /// <summary>
        /// update
        /// </summary>
        /// <param name="model"></param>
        /// <param name="header"></param>
        /// <returns></returns>
        [HttpPost]
        [Route("update")]
        public async Task<ActionResult> UpdateAsync([FromBody] CaseTypeModel model, [FromHeader] LocationHeader header)
        {
            model = (CaseTypeModel)EmptyFilter.Handler(model);
            var response = await this.caseTypeService.UpdateAsync(model);
            if (response > 0)
            {
                var auditLog = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.CaseType,
                    LogDate = DateTime.Now,
                    LogFrom = short.Parse(model.LoginRoleId.ToString()),
                    LogDescription = $"{model.CreatedByName} has updated <b>CaseType Name</b> of <strong>{model.CaseTypeName}</strong> successfully.",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLog);
            }

            return this.Success(response);
        }

        /// <summary>
        /// Fetches all asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("fetch-all")]
        public async Task<ActionResult> FetchAsync([FromBody] CaseTypeModel model)
        {
            model = (CaseTypeModel)EmptyFilter.Handler(model);
            var response = await this.caseTypeService.FetchAllAsync(model);
            return this.Success(response);
        }

        [HttpPost]
        [Route("modify-status")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ModifyStatusAsync([FromBody] CaseTypeModel model, [FromHeader] LocationHeader header)
        {
            model = (CaseTypeModel)EmptyFilter.Handler(model);
            var response = await this.caseTypeService.ActivateOrDeactivateTest(model);
            if (response == 0)
            {
                return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.CaseType,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $@"<b>{model.ModifiedByName}</b> has <b>{((bool)model.Active ? "Activated" : "Deactivated")}</b> the <b>{model.CaseTypeName}</b> successfully",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success(response);

        }

    }
}

